;********************************************************************************
;**
;** datagen.asm (Daten Generator)
;**
;** Description
;**
;** * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;**
;** Schedule:
;** 
;** Central requests a device (P+0x040=GA)
;** Device (PC-If or Fahrpult) answers with a request for func status (now the loc address is known) (0xE3 0x07 AdrH AdrL) 
;** DataGen answers with the func info of the addressed loco (P+0x60+GA 0xE3 0x50 S0 S1)
;**
;** Central requests a device (P+0x040=GA)
;** Device (PC-If or Fahrpult) answers with a cmd to set function status for a loco (0xE4 0x24 AH AL Gr1)
;** (Device (PC-If or Fahrpult) answers with a cmd to set function status for a loco (0xE4 0x25 AH AL Gr2))
;** (Device (PC-If or Fahrpult) answers with a cmd to set function status for a loco (0xE4 0x26 AH AL Gr3))
;**
;** ************************************************
;**
;** Central requests a device (P+0x040=GA)
;** Device (PC-If or Fahrpult) answers with a request for loco info (now the loc address is known) (0xE3 0x00 AdrH AdrL) 
;** DataGen answers with the info of the addressed loco (P+0x60+GA 0xE4 Kennung Speed F1 F2)
;**
;** **************************************
;**
;** Central requests a device (P+0x040=GA)
;** Device (PC-If or Fahrpult) answers with a cmd to control a loco (0xE4 0x10 AH AL RV)  14FS
;** (Device (PC-If or Fahrpult) answers with a cmd to control a loco (0xE4 0x11 AH AL RV)  27FS)
;** (Device (PC-If or Fahrpult) answers with a cmd to control a loco (0xE4 0x12 AH AL RV)  28FS)
;** (Device (PC-If or Fahrpult) answers with a cmd to control a loco (0xE4 0x13 AH AL RV) 128FS)
;** DataGen sends (unrequested) info to the device which did lock the loco before (P+0x60+GA 0xE3 0x40 AH AL)
;** (if the same device has addressed the loco or the loco was not occupied send a dummy Rufbyte, 
;** a dummy header and a XOR-Byte of kind: LI-information
;** RufByte:
;** Header: 
;** XOr:    
;** )
;**
;** **************************************
;**
;** Central requests a device (P+0x040=GA)
;** Device (PC-If or Fahrpult) answers with a cmd to set functions for a loco (0xE4 0x20 AH AL Gr1)
;** (Device (PC-If or Fahrpult) answers with a cmd to set functions for a loco (0xE4 0x21 AH AL Gr2))
;** (Device (PC-If or Fahrpult) answers with a cmd to set functions for a loco (0xE4 0x22 AH AL Gr3))
;** DataGen sends (unrequested) info to the device which did lock the loco before (P+0x60+GA 0xE3 0x40 AH AL)
;** (if the same device has addressed the loco or the loco was not occupied send a dummy Rufbyte, 
;** a dummy header and a XOR-Byte of kind: LI-information
;** RufByte:
;** Header: 
;** XOr:    
;** )
;**
;** **************************************
;**
;** In a first SRAM area the addresses of the locos and their "age" are stored. 
;** Depending on the value Tab4Length set in common.asm up to 32 locos are handled.
;**
;** Structure of Tab4 SRAM area:
;**
;** Tab4: Offset is SRAMTab4
;**
;** Offset+0   +1         +2         +3         +4        ...  +31
;** LocoAdr12  LocoAdr4   LocoAdr9   LocoAdr3   LocoAdr2       LocoAdr15 <--- LocoAddress
;** Offset+32  +33        +34        +35        +36       ...  +63
;** 31         19         1          6          2              7         <--- Age of loco address
;**
;** **************************************************
;**
;** In a second SRAM area the loco data are stored for 256 loco addresses
;**
;** SRAM structure:
;**
;** Tab3: Offset is SRAMTab3
;**
;** 0x0000 Data0       Data1       Data2       Data3       Data4       Data5       Data6       Data7
;**        0000 BFFF   RVVV VVVV   000F FFFF   FFFF FFFF   000S SSSS   SSSS SSSS   P11P PPPP   DB00 0000
;**
;** 0x0008 Data8       Data9       Data10      Data11      Data12      Data13      Data14      Data15
;**        Address     Speed       F1          F2          F3          F4          Special-F   U00D CCCC
;**
;** Data0 : Besetzt and number of speed steps
;** Data1 : Direction and Speed
;** Data2 : Actual Status of Functions (F0 F4F3F2F1)
;** Data3 : Actual Status of Functions (F12..F5)
;** Data4 : Status of Function behaviour (S0 S4S3S2S1) (1 = tastend)
;** Data5 : Status of Function behaviour (S12..S5)     (1 = tastend)
;** Data6 : This is the Rufbyte to the device which did the last access to this loco
;** Data7 : D: always set, if direction has changed, B: address of loco is in Tab4
;** Data8 : Address-Pattern
;** Data9 : Speed  -Pattern
;** Data10: F1     -Pattern
;** Data11: F2     -Pattern
;** Data12: F3     -Pattern
;** Data13: F4     -Pattern
;** Data14: Special-Pattern
;** Data15: U: Update-Bit; D: Direction (1=b, 0=f), CCCC: Counter for repeating direction change
;**
;********************************************************************************
;**
;** ### this three hashes indicate that something can be added to the source code
;**
;********************************************************************************

.include "../common/8515def.inc"

.equ BASpeedTable   = 0x10      ; Base address for Speed    Table   auf diese Adressen wird mit dem 
.equ BAFTable       = 0x20      ; Base address for Function Table   lpm-Kommando zugegriffen.
.equ BAAddressTable = 0x40      ; Base address for Address  Table 

;----------------------------------------------------------------------------------------------------
; SRAM-locations for data handling to X-Bus

.equ RxCntSRAM       = 0x0060 ; actual receive-XBus-data-counter
.equ NoOfDataSRAM    = 0x0061 ; first byte of received data contains the number of data-bytes  
                              ; + 2 (Header byte and XOr byte) to be received. This number is derived from the  
                              ; lower nibble of the first received byte (HEADER-Byte) 
.equ RxDataSRAM      = 0x0062 ; location where the first byte (header) of received data is stored 
                              ; 15 bytes (1 nibble) are possible to be received + 1 XOR byte which 
                              ; are stored at SRAM-Addr: 63,64,65,66,67,68,69,6a,6b,6c,6d,6e,6f,70,71,72 
.equ RufByteSRAM     = 0x0073 ; this is the location of the RufByte in SRAM
;----------------------------------------------------------------------------------------------------
.equ TxCntSRAM       = 0x0075 ; counts the transmitted data to X-Bus
.equ NoOfTxDataSRAM  = 0x0076 ; 
.equ TxHeaderSRAM    = 0x0077 ; SRAM location for Header byte to be transmiited to X-Bus
.equ TxDataSRAM      = 0x0078 ; 78,79,7a,7b,7c,7d,7e,7f,80,81,82,83,84,85,86,87,88,89

;----------------------------------------------------------------------------------------------------
.equ Tab3Data0SRAM   = 0x008A ; hold specific Tab3 Data0 value (evaluated from command)
.equ DefFSRAM        = 0x008B ; hold information which function group has to be modified

.equ SRAMTab4        = 0x0120 ; actual pattern for each device that controls a loco
.equ SRAMTab4End     = 0x015F ;

.equ SRAMTab3        = 0x0260 ; base address of table 3 (external SRAM for loco data)
.equ SRAMTab3End     = 0x124F ; last address of table 3 (256 x 16 byte = 4096 bytes)

.include "../common/common.asm"

;################################################################################
                                  ;                                           ###
.equ DeviceAddress   = DGenAddress ; defines the address of this device (DGen)###
                                  ; this address must be unique               ###
;################################################################################

;---------------------------------------------------------------------------------------------------- 
;.equ BaudRate        = 3     ; 62500 BAUD bei 4MHz  baud-rate for data-transmission 
.equ BaudRate        = 7          ; 62500 BAUD bei 8MHz  baud-rate for data-transmission 
;---------------------------------------------------------------------------------------------------- 
; registers and constants 
;---------------------------------------------------------------------------------------------------- 
.equ RelaisEn        = 5          ; PD5 Relais on/off for data to booster
.equ RS485DEX        = 4          ; PD4 DE for X-Bus-Net
.equ ShortCircuit    = 2          ; PD2 Short Circuit detect from booster (via at90s2313)

.equ MaxBits         = 18         ; Anzahl der zu versendenden Bits
     				  
.equ LocoAdrMax      = Tab4Length-1 ; maximale Anzahl von Lok-Adressen im Refresh-Zyklus
.equ SeqMax          =  3         ; es gibt 4 Sequenzen neben Speed: F1,F2,F3,F4
     				  
.equ LowerLimit      =  0         ; minimale Geschwindigkeit
.equ UpperLimit      = 14         ; maximale Geschwindigkeit
     				  
.equ AddrHigh        = 79         ; groesste Adresse
.equ AddrLow         =  0         ; kleinste Adresse

;-- SEG --------------------------------------------------
;.equ T1    	     = 464	      ; 58		   fuer 8MHz Teilbit Eins
;.equ T2    	     = 928        ; 116 ms    fuer 8MHz Teilbit Null


;-- /SEG -------------------------------------------------
.equ Short 	     = 208        ; 26 us      fuer 8MHz
.equ Long  	     = 1456       ; 7 x 26 us  fuer 8MHz
.equ T1    	     = 13120      ; 1.64 ms    fuer 8MHz
.equ T2    	     = 33920      ; 4.24 ms    fuer 8MHz
.equ T3    	     = 48800      ; 6.1 ms     fuer 8MHz
	  
.equ Idle  	     = 0b10101010 ; high-Byte fuer das Idle-Pattern

;----------------------------------------------------------------------------------------------------
; Sonderpattern fuer die Funktionstasten bei speziellen Geschwindigkeiten
;----------------------------------------------------------------------------------------------------

.equ PF1_2  = 0b11110001        ; Sonderpattern fuer Funktionstaste F1 bei Speed  2 (on)
.equ PF1_10 = 0b10110011        ; Sonderpattern fuer Funktionstaste F1 bei Speed 10 (on)
.equ PF2_3  = 0b00001101        ; Sonderpattern fuer Funktionstaste F2 bei Speed  3 (on)
.equ PF2_11 = 0b00011011        ; Sonderpattern fuer Funktionstaste F2 bei Speed 11 (on)
.equ PF3_5  = 0b00111101        ; Sonderpattern fuer Funktionstaste F3 bei Speed  5 (on)
.equ PF3_13 = 0b00111011        ; Sonderpattern fuer Funktionstaste F3 bei Speed 13 (on)
.equ PF4_6  = 0b11111101        ; Sonderpattern fuer Funktionstaste F4 bei Speed  6 (on)
.equ PF4_14 = 0b10111011        ; Sonderpattern fuer Funktionstaste F4 bei Speed 14 (on)

;----------------------------------------------------------------------------------------------------
; Sonderpattern fuer die Umschaltfunktion der Fahrtrichtung
;----------------------------------------------------------------------------------------------------

;.equ SpeedRB    = 0b11000101    ; -r 0xc5 dec197 
;.equ SpeedRF    = 0b10010001    ; +r 0x91 dec145  
.equ SpeedRB    = 0b11000000    ; -r c5
.equ SpeedRF    = 0b11000000    ; +r 91

.equ F1ROff     = 0b11010000    ;  r d0
.equ F2ROff     = 0b10000100    ;  r 84
.equ F3ROff     = 0b10010100    ;  r 94
.equ F4ROff     = 0b11010100    ;  r d4

;-- flags and positions in flag register
;-- FlagReg

.equ SendRequest        = 0    ; this bit is set, when Request-answer is transmitted to central
.equ XDataReceived      = 1    ; set, if a new byte was received from the X-Bus
.equ RufByteReceived    = 2    ; set if there was no data received from central before
.equ RufByteChecked     = 3    ; set after the RufByte was checked and is valid
.equ HeaderByteReceived = 4    ; this bit is set, when the first data after the RufByte was received
.equ SendTxData         = 6    ; set, if new data has to be send
.equ ShortCircuitDetect = 7    ; set, if a short circuit on the track was detected

;-- FlagReg2

.equ WaitQuit           = 2

;-------------------------

.def DeviceAdrTab3      = r1   ; device address from Tab3 Data6
.def LocoAdrLowTab1     = r2   ; low  byte of the loco address stored in Tab1 Data2
.def LocoAdrHighTab1    = r3   ; high byte of the loco address stored in Tab1 Data1

.def DeviceAdr          = r4   ; address of the X-Bus device
.def LocoAdrLow         = r5   ; low  byte of the loco address
.def LocoAdrHigh        = r6   ; high byte of the loco address

.def NrPause    	= r7   ; dieses Register zaehlt mit, welche Pause zu baearbeiten ist
.def LocoAdrPointer  	= r8   ; Zeiger auf die aktuell zu bearbeitende Loko-Adresse in Tab4
.def SeqPointer 	= r9   ; Zeiger auf die aktuell zu bearbeitende Sequenz (F1-F4)
.def SpPointer  	= r10  ; toggelt nur zwischen 0 und 1, wenn SpPointer '0' ist, dann ist immer
                	       ; die Geschwindigkeit zu bearbeiten, sonst die aktuelle Sequenz, auf die
                	       ; der SeqPointer zeigt
.def BitCnt     	= r11  ; zaehlt die zu sendenen Bits mit (0 bis MaxBits)
.def BitHalf    	= r12  ; bestimmt, welche Haelfte des zu sendenen Bits bearbeitet werden muss
                	       ; toggelt nur zwischen 0 und 1, wenn BitHalf '0' ist, dann ist immer
                	       ; die 1. Haelfte zu bearbeiten
.def DirChange  	= r13  ; gibt an, wenn >= 2, dass ein Richtungswechsel vorlag     
     	     		       
.def Tmp1               = r16  ; universel register
.def Tmp2               = r17  ; universel register
.def Tmp3               = r18  ; universel register
.def Tmp4               = r19  ; universel register
    
.def RxData             = r16  ;
.def TxData             = r16  ;
.def XBusCnt            = r17  ; counts number of received/transmitted bits
.def NoOfData           = r18  ; 

.def XOrByte            = r19  ;

.def RxDataTmp          = r20  ;

.def FlagReg            = r21  ; contains some flags
.def FlagReg2           = r22  ; contains further flags

.def HighByte   	= r23  ; oberste  8 Bit, die zu uebertragen sind
.def MiddleByte 	= r24  ; mittlere 8 Bit, die zu uebertragen sind
.def LowByte    	= r25  ; unterste 8 Bit, die zu uebertragen sind

;----------------------------------------------------------------------------------------------------
; Macros
;----------------------------------------------------------------------------------------------------

.macro   PUSHTmp
         push Tmp1           ; push tmp1 on the stack
         push Tmp2
         push Tmp3
         in   Tmp1,SREG      ; load status in tmp1
         push Tmp1           ; push tmp1 (status) on the stack
.endmacro

.macro   POPTmp
         pop  Tmp1           ; pop tmp1 (status) from stack
         out  SREG,Tmp1      ; write tmp1 (status) bach to status reg
         pop  Tmp3
         pop  Tmp2
         pop  Tmp1           ; pop tmp1 from stack
.endmacro

;----------------------------------------------------------------------------------------------------
; Interrupt table
;----------------------------------------------------------------------------------------------------
.cseg
.org 0
          rjmp  Reset                  ; jump to Reset-Handling
.org INT0addr
          rjmp  EXT_INT0               ; IRQ0 Handler (Short Circuit in track detect)
.org INT1addr
          reti                         ; not used in this application
.org ICP1addr
          reti                         ; not used in this application
.org OC1Aaddr                           
          reti                         ; not used in this application
.org OC1Baddr                           
          rjmp  TIM1_COMPB             ; jump to timer1 compare match 
.org OVF1addr   
          reti                         ; not used in this application
.org OVF0addr   
          reti                         ; not used in this application
.org SPIaddr   
          reti                         ; not used in this application
.org URXCaddr                          
          rjmp  UART_RXC               ; UART RX Complete Handle     
.org UDREaddr   
          reti                         ; not used in this application
.org UTXCaddr                          
          reti                         ; UART TX CompleteHandle
.org ACIaddr   
          reti                         ; not used in this application

;----------------------------------------------------------------------------------------------------

.cseg
.org 0x0100 ; start of program memory

;-------------------------------------------------------------------------------------------------------
;-- EEPROM Read-Routine
;-------------------------------------------------------------------------------------------------------

.def     EEdrd        = r0   ; for the EEPROM-read
.def     EEard        = r16  ; for the EEPROM-read

EERead:
        sbic  EECR,EEWE
        rjmp  EERead
        clr   Tmp2
        out   EEARH,Tmp2
        out   EEARL,EEard
        sbi   EECR,EERE
        in    EEdrd,EEDR
        ret

;----------------------------------------------------------------------------------------------------
;-- Set the Stackpointer

Reset:      ldi   Tmp1,low(RAMEND)
            out   SPL,Tmp1           ; init Stack Pointer
            ldi   Tmp1,high(RAMEND)
            out   SPH,Tmp1           ; init Stack Pointer

;----------------------------------------------------------------------------------------------------
;-- set the MCU Control Register - MCUCR

            ldi   Tmp1,1<<SRE          ; enable the external SRAM
            out   MCUCR,Tmp1

;----------------------------------------------------------------------------------------------------
;-- clear SRAM from 0x0060 to SRAMTab4End

ClearSRAM:  clr   Tmp1
            ldi   Tmp2,high(SRAMTab4End+1)
            ldi   Tmp3,low(SRAMTab4End+1)
            clr   ZH            ;  initialize z-pointer
            ldi   ZL,$60        ;  initialize z-pointer
DoMore:     st    Z+,Tmp1       ;  clear SRAM-location 
            cpse  ZH,Tmp2       ;  compare high-byte of z-pointer
            rjmp  DoMore        
            cpse  ZL,Tmp3       ;  compare low-byte of z-pointer
            rjmp  DoMore

;--------------------------------------------------------------------------------            

;-- set addresses in Tab3 Data8 to the Idle Pattern and clear specific RAM-locations

            ldi   Tmp1,Idle
            clr   Tmp2
            ldi   Tmp3,low(SRAMTab3End-15)
            ldi   Tmp4,high(SRAMTab3End-15)
            ldi   r20,0b10000000
            ldi   ZL,low(SRAMTab3-16)
            ldi   ZH,high(SRAMTab3-16)
SetIdle:    adiw  ZL,16
            std   Z+0,Tmp2
            std   Z+1,r20        ; RVVV VVVV
            std   Z+2,Tmp2
            std   Z+3,Tmp2
            std   Z+4,Tmp2
            std   Z+5,Tmp2
            std   Z+6,Tmp2
            std   Z+7,Tmp2
            std   Z+8,Tmp1
            std   Z+9,Tmp2
            std   Z+10,Tmp2
            std   Z+11,Tmp2
            std   Z+12,Tmp2
            std   Z+13,Tmp2
            std   Z+14,Tmp2
            std   Z+15,Tmp2

            cpse  ZL,Tmp3
            rjmp  SetIdle
            cpse  ZH,Tmp4
            rjmp  SetIdle

;--------------------------------------------------------------------------------
;-- set default addresses in Tab4

            ldi   Tmp1,11       ; (default loco address)
            ldi   Tmp2,32
            clr   Tmp3
            ldi   ZL,low(SRAMTab4)
            ldi   ZH,high(SRAMTab4)
SetAddress: inc   Tmp3
            st    Z+,Tmp1
            cpse  Tmp3,Tmp2
            rjmp  SetAddress

;-- set predefined addresses in Tab4
            ldi   ZL,low(SRAMTab4)
            ldi   ZH,high(SRAMTab4)
            ldi   Tmp1,11       ; BR01
            st    Z+,Tmp1
            ldi   Tmp1,60       ; Glaskasten
            st    Z+,Tmp1
            ldi   Tmp1,70       ; Personenzug
            st    Z+,Tmp1
            ldi   Tmp1,74       ; Hoellental
            st    Z+,Tmp1
            ldi   Tmp1,75       ; Nebenbahn Personenzug
            st    Z+,Tmp1

;-- set age of addresses from 1 to 32
            clr   Tmp1
            ldi   Tmp2,32
            ldi   ZL,low(SRAMTab4+32)
            ldi   ZH,high(SRAMTab4+32)
SetAge:     inc   Tmp1
            st    Z+,Tmp1
            cpse  Tmp1,Tmp2
            rjmp  SetAge

;-------------------------------------------------------------------------------- 
; set B bit in Tab3 Data7 for the specific addresses just defined in Tab4

            clr   LocoAdrHigh
            ldi   Tmp2,0b01000000 ; define B bit for Tab3 Data7 Bit6

            ldi   Tmp1,11       ; BR01
            mov   LocoAdrLow,Tmp1
            rcall Mul16Tab3
            std   Z+7,Tmp2

            ldi   Tmp1,74       ; Hoellental
            mov   LocoAdrLow,Tmp1
            rcall Mul16Tab3
            std   Z+7,Tmp2

            ldi   Tmp1,75       ; Nebenbahn Personenzug
            mov   LocoAdrLow,Tmp1
            rcall Mul16Tab3
            std   Z+7,Tmp2

;----------------------------------------------------------------------------------------------------
;-- Initialize all registers and SRAM-locations

Init:       
            ldi   r16,30
            clr   r0
            mov   r1,r16
            ldi   ZL,2
            clr   ZH
ClrGPReg:   st    Z+,r0
            cpse  ZL,r1
            rjmp  ClrGPReg

            ldi   XH,high(RxDataSRAM)  ; restore X-pointer for getting new data from X-Bus
            ldi   XL,low(RxDataSRAM)

            ldi   HighByte,Idle        ; Idle-Pattern
            clr   MiddleByte           ; Idle-Pattern
            clr   LowByte              ; Idle-Pattern

;----------------------------------------------------------------------------------------------------
;--------------------
;-- define ports
;--------------------

;-- PA7  : Address(low)/Data to external SRAM 
;-- PA6  : Address(low)/Data to external SRAM 
;-- PA5  : Address(low)/Data to external SRAM 
;-- PA4  : Address(low)/Data to external SRAM 
;-- PA3  : Address(low)/Data to external SRAM 
;-- PA2  : Address(low)/Data to external SRAM 
;-- PA1  : Address(low)/Data to external SRAM 
;-- PA0  : Address(low)/Data to external SRAM 
;--
;-- PB7  :
;-- PB6  :
;-- PB5  :
;-- PB4  : used for test-LEDs
;-- PB3  : used for test-LEDs
;-- PB2  : used for test-LEDs
;-- PB1  : used for test-LEDs
;-- PB0  : used for test-LEDs
;--
;-- PC7  : Address(high) to external SRAM 
;-- PC6  : Address(high) to external SRAM 
;-- PC5  : Address(high) to external SRAM 
;-- PC4  : Address(high) to external SRAM 
;-- PC3  : Address(high) to external SRAM 
;-- PC2  : Address(high) to external SRAM 
;-- PC1  : Address(high) to external SRAM 
;-- PC0  : Address(high) to external SRAM 
;--
;-- PD7  :
;-- PD6  :
;-- PD5  : Output Relais
;-- PD4  : Output DE to enable RS485 transmitter (to X-Bus)
;-- PD3  : 
;-- PD2  : Input  Int0 (Short-Circuit detect from at90s2313)
;-- PD1  : Output Tx for Hardware-UART
;-- PD0  : Input  Rx for Hardware-UART (PullUp)
;-- 

            ldi   Tmp1,0b00000000      ; data-direction for PortA ('1' = Output)
            out   DDRA,Tmp1            ; 
            ldi   Tmp1,0b00000000      ; AD0-7 (address and data for ext. SRAM)
            out   PORTA,Tmp1           ;

            ldi   Tmp1,0b11111111      ; data-direction for PortB ('1' = Output)
            out   DDRB,Tmp1   
			         ; *********************************************
            ldi   Tmp1,0b11101100      ; LEDs are switched to VCC, "1" is off
            out   PORTB,Tmp1           ;

            ldi   Tmp1,0b00000000      ; data-direction for PortC ('1' = Output)
            out   DDRC,Tmp1            ; 
            ldi   Tmp1,0b00000000      ; A8-15 (address for ext. SRAM)
            out   PORTC,Tmp1           ;

            ldi   Tmp1,0b11110010      ; data-direction for PortD ('1' = Output)
            out   DDRD,Tmp1            ;
            ldi   Tmp1,0b00000011      ; clr WR_n and RD_n, clr RelaisEn, clr RS485DEX (to X-Bus) => receiver enabled
            out   PORTD,Tmp1           ; PD0=RxD, PD1=TxD, PD4=RS485DEX, PD6=WR_n, PD7=RD_n

;----------------------------------------------------------------------------------------------------
;-- Timer0 and Timer1 interrupt flags

;----------------------------------------------------------------------------------------------------
;--------------------------------------------------------------------------------
;-- Timer1

            ldi   Tmp1,0b00100000      ; enable den Interrupt fuer Compare B
            out   TIMSK,Tmp1
 
            ldi   Tmp1,0b00000000      ; lade das Vergleichsregister fuer A mit einem kleinen Startwert
            out   OCR1BH,Tmp1          ; damit wird erreicht, dass die Interruptroutine mit dem Start-
            ldi   Tmp1,0b00000100      ; wert '1' fuer den Ausgang beginnt.
            out   OCR1BL,Tmp1

            ldi   Tmp1,0b00010000      ; toggle beim naechsten Interrupt den Ausgang OCR1B
            out   TCCR1A,Tmp1
            
;--------------------------------------------------------------------------------
 
                                       ; mit diesem Register wird auch der Clock-Prescaler definiert
                                       ; Sobald hier ein Wert != 0 gesetzt wird, faengt der Zaehler
                                       ; an zu zaehlen (vorwaerts)
                                       ; tritt Gleichheit auf, so wird ein Interrupt ausgeloest
            ldi   Tmp1,1               ; Prescaler steht auf 1024 mit 101
            out   TCCR1B,Tmp1          ; Prescaler steht auf    1 mit 001
                    
;----------------------------------------------------------------------------------------------------
;-- enable the hw-uart

            ldi   Tmp1,BaudRate        ; set UART-BAUT-Register (p. 3-47)
            out   UBRR,Tmp1            ;
            sbi   UCR,RXCIE 
            sbi   UCR,RXEN             ; set UART-CONTROL-REGISTER (p. 3-45)
            sbi   UCR,TXEN             ;

;----------------------------------------------------------------------------------------------------
;-- enable the global interrupt

            in    Tmp1,MCUCR           ; read IO because of ext SRAM bit
            ori   Tmp1,0b00000011
            out   MCUCR,Tmp1           ; On rising edges
            ldi   Tmp1,0b01000000
            out   GIMSK,Tmp1           ; Enable external interrupt 0
            ser   Tmp1
            out   GIFR,Tmp1            ; clear possibly set bits which can cause an interrupt
                                       ; after enabling the interrupts
  
;----------------------------------------------------------------------------------------------------
;-- enable the global interrupt

            sei                        ; enable Interrupts in Status Register
  
;----------------------------------------------------------------------------------------------------

;----------------------------------------------------------------------------------------------------
;-- Mainprogram (only flags are checked in this loop)
;----------------------------------------------------------------------------------------------------

Main:   
            sbrc  FlagReg,XDataReceived    ; new byte was received from X-Bus device?
            rcall GetCData                 ; yes than jump
	    sbrc  FlagReg,SendTxData       ; should data be send?
	    rcall TxDataSend               ; yes than jump
            rjmp  Main

;----------------------------------------------------------------------------------------------------
;-- Subroutine which checks the incoming data from any device on the XBus
;--
;-- New byte was received from the central
;-- check if a complete command was received
;-- check if command contains no errors (parity-check and XOr-check)
;-- react on command if it is valid
;----------------------------------------------------------------------------------------------------

GetCData:
            cbr   FlagReg,1<<XDataReceived; clear flag now, because new data will be evaluated now
            sbrs  FlagReg,RufByteChecked; if this bit is cleared, than the received byte must be the RufByte
            rjmp  CheckRufByte          ; which will be checked now

                                        ; if the bit is set, than the Rufbyte was always received and checked
                                        ; and the new byte must be a data (Header, data or XOR) byte

            sbrs  FlagReg,HeaderByteReceived ; is Header Byte Receive-Flag just set?
            rjmp  CheckHeaderByte       ; no

            ; if the HeaderByteReceived-flag is set the new received byte can only be a data or the XOr byte
            ; to check this the number of bytes from the header-byte will be checked against the actual 
            ; X-Bus data-counter

            lds   XBusCnt,RxCntSRAM     ; reload the X-Bus data-counter from SRAM
            lds   NoOfData,NoOfDataSRAM ; reload the lower nibble of the commands header byte from SRAM
            cp    XBusCnt,NoOfData      ; ask if all bytes from command are received
            breq  CheckXCmd             ; yes, than check now the command (XOr-check)

            ; no, not all data were received so far

            inc   XBusCnt               ; increment the central data-counter
            sts   RxCntSRAM,XBusCnt     ; and store it back to the SRAM
            ret                         ; go back to main-loop
            
            ; all data are received, do now the XOR-check

CheckXCmd:  clr   XBusCnt               ; clear the Rx-data-counter
            sts   RxCntSRAM,XBusCnt     ; and store value in SRAM
            ldi   ZH,high(NoOfDataSRAM) ; load Z-Register with first address of SRAM space where
            ldi   ZL,low(NoOfDataSRAM)  ; the rx data is stored
            ld    NoOfData,Z+           ; get number of data from SRAM
            subi  NoOfData,-2           ; add 2 because of Header and XOR-Byte
            ld    RxData,Z+             ; get first data from SRAM (Header Byte)
IncXBusCnt: inc   XBusCnt               ; increment the X-Bus-Rx-data-counter
            cp    XBusCnt,NoOfData      ; compare it with the maximum number of data in SRAM
            breq  XORComplete           ; if equal, XORed data must be 0 if no erroro occured
            ld    RxDataTmp,Z+          ; if not, get next data from SRAM
            eor   RxData,RxDataTmp      ; XOR new data with all XORed data before
            rjmp  IncXBusCnt            ; loop

            ; the following flags in FlagReg can be now cleared anyway
XORComplete:cbr   FlagReg,(1<<RufByteReceived)|(1<<RufByteChecked)|(1<<HeaderByteReceived)
            ldi   XH,high(RxDataSRAM)   ; restore the X-pointer to be ready for receiving new 
            ldi   XL,low(RxDataSRAM)    ; data from the X-Bus devices
            cpi   RxData,0              ; check if error occured
            breq  NoComError            ; XOR is correct, so there was no communication error
            ret                         ; ### add error alarm routine here (alarm 1)

NoComError:
            ; XOr checked passed
            ; ask now, if the Rufbyte did has addressed this device

            lds   Tmp1,RufByteSRAM      ; read RufByte from SRAM
            andi  Tmp1,0b00011111       ; mask only the GA (Geraete-Adresse)
            cpi   Tmp1,0                ; ask if command was a BroadCast (BC)
            breq  XCmdIsBC              ; yes command was a BC
            cpi   Tmp1,DeviceAddress    ; ask if RufByte addresses this device
            breq  XCmdValid             ; yes command is valid for this device and it is not a request

;-- Header Byte evaluation
            lds   Tmp1,RxDataSRAM
            cpi   Tmp1,0x92             ; eine Lok anhalten (only execute cmd)
            breq  Header0x92
            cpi   Tmp1,0xE3             ; Lokdaten oder Funktiosstatus anfordern (give feedback)
            breq  Header0xE3
            cpi   Tmp1,0xE4             ; Lok Fahrbefehl, Lok Funktionsstatus oder Funktionsstatus setzen (only execute cmd)
            breq  Header0xE4J
            ret                         ; ### add error alarm routine here (alarm 1)


XCmdValid:                              ; device was addressed by a P+0x60+GA 
            ret                         ; ### add function for the different kinds of Cmds

;--------------------
Header0xE4J:rjmp  Header0xE4
;--------------------

XCmdIsBC:   ;check now the kind of the the BC
            ; Alles An       : 0x61 0x01 0x60
            ; Alles Aus      : 0x61 0x00 0x61
            ; Alle Loks aus  : 0x81 0x00 0x81
            ; Programmiermode: 0x61 0x02 0x63
            lds   Tmp1,RxDataSRAM       ; get header byte from SRAM
            cpi   Tmp1,0x61
            breq  Header0x61
            cpi   Tmp1,0x81
            breq  Header0x81
            ret                         ; ignore other cmds

Header0x61:
            lds   Tmp1,RxDataSRAM+1     ; get first data byte from SRAM
            cpi   Tmp1,0x01             ; Alles An
            breq  Data10x01
            cpi   Tmp1,0x00             ; Alles Aus
            breq  Data10x00
            ret                         ; ignore other cmds

Header0x81: 
            lds   Tmp1,RxDataSRAM+1     ; get first data byte from SRAM
            cpi   Tmp1,0x00             ; Alle Loks Aus
            breq  Data10x00a
            ret                         ; ignore other cmds

Data10x01:  ; Alles An
            sbis  PIND,ShortCircuit     ; check if short circuit is always present
            sbi   PortD,RelaisEn        ; no, than switch on the signal to the booster
            ret
Data10x00:  ; Alles Aus
Data10x00a: ; Alle Loks Aus
            cbi   PortD,RelaisEn        ; switch off the signal to the booster
            ret

;-- BC-cmds: end
;--------------------
; The following cmds are commands from any device which have to be evaluated by the DataGen device 

;-- Data 1 evaluation
Header0x92:                             ; eine Lok anhalten
            lds   Tmp1,RxDataSRAM+2     ; shift now the loco address to that position and fill all other
            sts   RxDataSRAM+3,Tmp1     ; values with data which are similar to the command "Set Loco Speed"
            lds   Tmp1,RxDataSRAM+1
            sts   RxDataSRAM+2,Tmp1
            ldi   Tmp1,0x10             ; this is the Kennung (14 Speedsteps) ###
            sts   RxDataSRAM+1,Tmp1
            ldi   Tmp1,0x00             ; this is the direction and speed (forward and stop)
            sts   RxDataSRAM+4,Tmp1
            rjmp  Header0xE4            ; continue like setting speed 0 for the selected loco

;--------------------------------------------------

Header0xE3: lds   Tmp1,RxDataSRAM+1     ; get first data byte from SRAM
            cpi   Tmp1,0x00             ; Lokdaten anfordern
            breq  Req4LocoData
            cpi   Tmp1,0x07             ; Funktionsstatus anfordern
            breq  Req4FuncData         
            ret                         ; ignore other cmds

Req4LocoData: 
            ;question was (0xE3 0x00 AdrH AdrL) 
            ;answer is    (P+0x60+GA 0xE4 Kennung Speed F1 F2)     
            lds   LocoAdrLow,RxDataSRAM+3
            lds   LocoAdrHigh,RxDataSRAM+2
            rcall Mul16Tab3

	    lds   TxData,RufByteSRAM    ; RufByte
            ori   TxData,0x60
            ldi   Tmp2,0x80
            eor   TxData,Tmp2
	    sts   TxDataSRAM,TxData
            mov   Tmp2,TxData           ; store new Rufbyte in Tmp2 for checking it against Tab3 Data6

            ldi   XOrByte,0xE4          ; HeaderByte
            sts   TxDataSRAM+1,XOrByte
            ldd   Tmp3,Z+6              ; load now Tab3 Data6 "P11P PPPP"
            ld    TxData,Z+             ; Kennung
            sbrs  TxData,3              ; is Tab3 Data0 B bit set for this loco address?
            rjmp  AdrFree               ; no, this loco address is free
            ; because the bit is set, check now, if the same device which controls the loco did ask
            ; for the loco info. In this case, do not set the B-Bit in the answer
            cp    Tmp2,Tmp3             ; compare the two request-bytes
            brne  AdrFree
            andi  TxData,0b11110111     ; it is the same device -> clear the bit
AdrFree:    eor   XOrByte,TxData
            sts   TxDataSRAM+2,TxData
            ld    TxData,Z+             ; Speed
            eor   XOrByte,TxData
            sts   TxDataSRAM+3,TxData
            ld    TxData,Z+             ; F1
            eor   XOrByte,TxData
            sts   TxDataSRAM+4,TxData
            ld    TxData,Z+             ; F2
            eor   XOrByte,TxData
            sts   TxDataSRAM+5,TxData
            sts   TxDataSRAM+6,XOrByte  ; XOrByte
            ldi   TxData,7
            sts   NoOfTxDataSRAM,TxData ; store number of transmitted data to SRAM
            sbr   FlagReg,1<<SendTxData ; set flag to indicate that data has to be send 
            ret

Req4FuncData:
            ;question was (0xE3 0x07 AdrH AdrL) 
            ;answer is    (P+0x60+GA 0xE3 0x50 S0 S1)
            lds   LocoAdrLow,RxDataSRAM+3
            lds   LocoAdrHigh,RxDataSRAM+2
            rcall Mul16Tab3

	    lds   TxData,RufByteSRAM    ; RufByte
            ori   TxData,0x60
            ldi   Tmp2,0x80
            eor   TxData,Tmp2
	    sts   TxDataSRAM,TxData

            ldi   XOrByte,0xE3          ; HeaderByte
            sts   TxDataSRAM+1,XOrByte
            ldi   TxData,0x50           ; 0x50
            eor   XOrByte,TxData
            sts   TxDataSRAM+2,TxData
            adiw  ZL,4                  ; S0 is located at Z+4, S1 at Z+5
            ld    TxData,Z+             ; S0
            eor   XOrByte,TxData
            sts   TxDataSRAM+3,TxData
            ld    TxData,Z+             ; S1
            eor   XOrByte,TxData
            sts   TxDataSRAM+4,TxData
            sts   TxDataSRAM+5,XOrByte  ; XOrByte
            ldi   TxData,6
            sts   NoOfTxDataSRAM,TxData ; store number of transmitted data to SRAM
            sbr   FlagReg,1<<SendTxData ; set flag to indicate that data has to be send 
            ret

;--------------------------------------------------

Header0xE4: lds   Tmp1,RxDataSRAM+1     ; get first data byte from SRAM

            lds   LocoAdrHigh,RXDataSRAM+2; load new loco address from Rx-data area   
            lds   LocoAdrLow,RXDataSRAM+3; load new loco address from Rx-data area   

            cpi   Tmp1,0x10             ; Lok Fahrbefehl (0x10,0x11,0x12,0x13)
            breq  SetLocoSpeed14
            cpi   Tmp1,0x11             ; Lok Fahrbefehl (0x10,0x11,0x12,0x13)
            breq  SetLocoSpeed27
            cpi   Tmp1,0x12             ; Lok Fahrbefehl (0x10,0x11,0x12,0x13)
            breq  SetLocoSpeed28
            cpi   Tmp1,0x13             ; Lok Fahrbefehl (0x10,0x11,0x12,0x13)
            breq  SetLocoSpeed128

            rjmp  SetLocoFunc

SetLocoSpeed14:
            ldi   Tmp1,0b00001000        ; set Tab3 Data0 here
            rjmp  SetLocoSpeed
SetLocoSpeed27:
            ldi   Tmp1,0b00001001        ; set Tab3 Data0 here
            rjmp  SetLocoSpeed
SetLocoSpeed28:
            ldi   Tmp1,0b00001010        ; set Tab3 Data0 here
            rjmp  SetLocoSpeed
SetLocoSpeed128:
            ldi   Tmp1,0b00001100        ; set Tab3 Data0 here
            rjmp  SetLocoSpeed

SetLocoSpeed:
            sts   Tab3Data0SRAM,Tmp1    ; store 0b0000BFFF. This value is needed later
            lds   DeviceAdr,RufByteSRAM ; load the Rufbyte from SRAM. This RufByte contains the address
                                        ; of the device which sends this command.
            ldi   Tmp2,0b00011111
            and   DeviceAdr,Tmp1        ; mask only the device address

            rcall Mul16Tab3
            ld    Tmp2,Z                ; load Tab3 Data0
           
;--------------------------------------------------
;-- 
            sbrs  Tmp2,3                ; is Tab3 Data0 B bit set for this loco address?
            rjmp  SCase2                ; no it is not. Proceed now with case nbr 2
            
SCase1:     st    Z,Tmp1                ; store Tab3 Data0
            ldd   Tmp3,Z+7              ; read Tab3 Data7 for reading Bit7
            andi  Tmp3,0b01111111       ; clear direction-has-changed Bit7 in Tab3 Data7
            lds   Tmp1,RxDataSRAM+4     ; get new speed and direction info from rx data area
            ldd   Tmp2,Z+1              ; read old direction info to compare it with the new direction info
            eor   Tmp2,Tmp1             ; did direction changed?
            sbrc  Tmp2,7                ; no
            ori   Tmp3,0b10000000       ; yes, set Bit7 in Tab3 Data7
            std   Z+7,Tmp3              ; and store info to Tab3 Data7 
            std   Z+1,Tmp1              ; store new speed and direction back to Tab3 Data1
            ldd   DeviceAdrTab3,Z+6     ; load Rufbyte for BC-Info to device which controlled the loco till now
            lds   Tmp1,RufByteSRAM      ; get now the Rufbyte for the device that sends this command actualy
            ori   Tmp1,0x60             ; this is the code for the BC (bits 5 and 6)
            ldi   Tmp2,0x80             ; because one bit is set more in the Rufbyte change the parity bit
            eor   Tmp1,Tmp2             ; this is done here
            std   Z+6,Tmp1              ; store Tab3 Data6 with new calculated Rufbyte
            cp    DeviceAdrTab3,Tmp1    ; check if the device is the same that controlled the loco before
            breq  SSameDev              ; if yes, than send LI-message only  
            rcall BC2OldDev             ; put all data to Tx-RAM area to send BC to old device
            rcall UpdateTab3            ; do now the speed and function pattern actualization
            rcall UpdateTab4            ; actualize now Tab4
            ret
            
SCase2:     st    Z,Tmp1                ; store to Tab3 Data0 
            ldd   Tmp3,Z+7              ; read Tab3 Data7 for reading Bit7
            andi  Tmp3,0b01111111       ; clear direction-has-changed Bit7 in Tab3 Data7
            lds   Tmp1,RxDataSRAM+4     ; get new speed and direction info from rx data area
            ldd   Tmp2,Z+1              ; read old direction info to compare it with the new direction info
            eor   Tmp2,Tmp1             ; did direction changed?
            sbrc  Tmp2,7                ; no
            ori   Tmp3,0b10000000       ; yes, set Bit7 in Tab3 Data7
            std   Z+7,Tmp3              ; clear any info about direction-has-changed now in Tab3 Data7
            std   Z+1,Tmp1              ; store new speed and direction back to Tab3 Data1
            lds   Tmp1,RufByteSRAM      ; get now the Rufbyte for the device that sends this command
            ori   Tmp1,0x60             ; this is the code for the BC (bits 5 and 6)
            ldi   Tmp2,0x80             ; because one bit is set more in the Rufbyte change the parity bit
            eor   Tmp1,Tmp2             ; this is done here
            std   Z+6,Tmp1              ; store Tab3 Data6 with new calculated Rufbyte
            mov   DeviceAdrTab3,Tmp1    ; move the new Rufbyte from Tmp1, because the following sub-routine
SSameDev:   rcall BC2SameDev            ; needs it in an other register
            rcall UpdateTab3            ; do now the speed and function pattern actualization
            rcall UpdateTab4            ; actualize now Tab4
            ret

; --------------------------------------------------

SetLocoFunc:cpi   Tmp1,0x20             ; Lok Funktionsbefehl (0x20,0x21,0x22)
            breq  SetLocoFuncGr1
            cpi   Tmp1,0x21             ; Lok Funktionsbefehl (0x20,0x21,0x22)
            breq  SetLocoFuncGr2
            cpi   Tmp1,0x22             ; Lok Funktionsbefehl (0x20,0x21,0x22)
            breq  SetLocoFuncGr3

            rjmp  SetFunc

SetLocoFuncGr1:
            ldi   Tmp1,0x02             ; set Tab3 Data2
            rjmp  SetLocoFunct
SetLocoFuncGr2:   ret
            ldi   Tmp1,0x03             ; set Tab3 Data3 (low nibble)
            rjmp  SetLocoFunct
SetLocoFuncGr3:   ret
            ldi   Tmp1,0x13             ; set Tab3 Data3 (high nibble)
            rjmp  SetLocoFunct

SetLocoFunct:
            sts   DefFSRAM,Tmp1         ; store information which byte/nibble has to be stored later
            lds   DeviceAdr,RufByteSRAM ; load the Rufbyte from SRAM. This RufByte contains the address
                                        ; of the device which sends this command.
            ldi   Tmp1,0b00011111
            and   DeviceAdr,Tmp1        ; mask only the device address

            rcall Mul16Tab3
            ld    Tmp2,Z                ; load Tab3 Data0
           
;--------------------------------------------------
;-- 
            sbrs  Tmp2,3                ; is Tab3 Data0 B bit set for this loco address?
            rjmp  FCase2                ; no it is not. Proceed now with case nbr 2

FCase1:
            ; Tab3 Data0 B-bit just set (case 2)
            rcall FUpdate               ; in this routine Tab3 Data2 or Data3 (High or Low nibble) are updated

            ldd   DeviceAdrTab3,Z+6     ; load Rufbyte for BC-Info to device which controlled the loco until now
            lds   Tmp1,RufByteSRAM      ; get now the Rufbyte for the device that sends this command
            ori   Tmp1,0x60             ; this is the code for the BC (bits 5 and 6)
            ldi   Tmp2,0x80             ; because one bit is set more in the Rufbyte change the parity bit
            eor   Tmp1,Tmp2             ; this is done here
            std   Z+6,Tmp1              ; store Tab3 Data6 with new calculated Rufbyte
            cp    DeviceAdrTab3,Tmp1    ; check if the device is the same that controlled the loco before
            breq  FSameDev              ; if yes, than send LI-message only  
;            mov   DeviceAdrTab3,Tmp2    ; restore DeviceAdrTab3
            rcall BC2OldDev             ; put all data to Tx-RAM area to send BC to old device
            rcall UpdateTab3            ; do now the speed and function pattern actualization
            rcall UpdateTab4            ; actualize now Tab4
            ret

FCase2:     ori   Tmp2,0b00001000       ; set B bit in Tab3 Data0 now
            st    Z,Tmp2                ; and store it back to Tab3 Data0 for the specified loco address

            rcall FUpdate               ; in this routine Tab3 Data2 or Data3 (High or Low nibble) are updated

            lds   Tmp1,RufByteSRAM      ; get now the Rufbyte for the device that sends this command
            ori   Tmp1,0x60             ; this is the code for the BC (bits 5 and 6)
            ldi   Tmp2,0x80             ; because one bit is set more in the Rufbyte change the parity bit
            eor   Tmp1,Tmp2             ; this is done here
            std   Z+6,Tmp1              ; store Tab3 Data6 with new calculated Rufbyte
            mov   DeviceAdrTab3,Tmp1    ; move the new Rufbyte from Tmp1, because the following sub-routine
FSameDev:   rcall BC2SameDev            ; needs it in an other register
            rcall UpdateTab3            ; do now the speed and function pattern actualization
            rcall UpdateTab4            ; actualize now Tab4
            ret

; --------------------------------------------------

SetFunc:    cpi   Tmp1,0x24             ; Funktionsstatus setzen (0x24,0x25,0x26)
            breq  SetFuncGr1              
            cpi   Tmp1,0x25             ; Funktionsstatus setzen (0x24,0x25,0x26)
            breq  SetFuncGr2              
            cpi   Tmp1,0x26             ; Funktionsstatus setzen (0x24,0x25,0x26)
            breq  SetFuncGr3              
            ret                         ; ignore other cmds

SetFuncGr1: ; cmd was: (0xE4 0x24 AH AL Gr1)
            rcall Mul16Tab3             ; calculate base-address for the loco
            lds   RxData,RxDataSRAM+4   ; get new Group 1-setting from Rx-Data-SRAM
            std   Z+4,RxData            ; and store it now in table 3
            ret

SetFuncGr2: ; cmd was: (0xE4 0x25 AH AL Gr2)
            rcall Mul16Tab3             ; calculate base-address for the loco
            ldd   Tmp2,Z+5              ; get group 2 and 3 settings from table 3
            andi  Tmp2,0xF0             ; clear lower nibble (group 2-settings)
            lds   RxData,RxDataSRAM+4   ; get new group 2-setting from Rx-Data-SRAM
            or    RxData,Tmp2           ; combine new group 2 and old group 3 setting again
            std   Z+5,RxData            ; and store it now in table 3
            ret

SetFuncGr3: ; cmd was: (0xE4 0x26 AH AL Gr3)
            rcall Mul16Tab3             ; calculate base-address for the loco
            ldd   Tmp2,Z+5              ; get group 2 and 3 settings from table 3
            andi  Tmp2,0x0F             ; clear upper nibble (group 3-settings)
            lds   RxData,RxDataSRAM+4   ; get new group 3-setting from Rx-Data-SRAM
            swap  RxData                ; group 3 is in the upper nibble of byte in table 3
            or    RxData,Tmp2           ; combine old group 2 and new group 3 setting again
            std   Z+5,RxData            ; and store it now in table 3
            ret

;--------------------------------------------------

.include "datagen_update_tab3.asm"
.include "datagen_update_tab4.asm"

;--------------------------------------------------

Mul16Tab3:  ; mutiply the loco address with 16 and add the offset for the table 3
            ; result is stored in Z-register

            mov   ZL,LocoAdrLow
            mov   ZH,LocoAdrHigh        ; LocoAdrHigh is 0xC0 OR 0xAdr
            andi  ZH,0b00111111         ; 0xC0 iss masked now

            lsl   ZL                    ; mult 2
            rol   ZH         
            lsl   ZL                    ; mult 4
            rol   ZH         
            lsl   ZL                    ; mult 8
            rol   ZH         
            lsl   ZL                    ; mult 16
            rol   ZH         

            subi  ZL,low(-SRAMTab3)     ; add offset for table 3 to result
            sbci  ZH,high(-SRAMTab3)
            ret

;-------------------

FUpdate:
            lds   Tmp1,RxDataSRAM+4     ; get new function info from rx data area

            lds   Tmp2,DefFSRAM         ;
            cpi   Tmp2,0x02
            breq  Add2
            cpi   Tmp2,0x03
            breq  Add3L

Add3H:      
            ldd   Tmp2,Z+3              ; reload function from Tab3 Data3 
            andi  Tmp2,0b00001111       ; clear upper nibble
            swap  Tmp1                  ; swap nibbles of new function
            or    Tmp2,Tmp1             ; add bits in upper nibble to the function
            std   Z+3,Tmp2              ; store updated function back to Tab3 Data3
            ret

Add3L:       
            ldd   Tmp2,Z+3              ; reload function from Tab3 Data3 
            andi  Tmp2,0b11110000       ; clear lower nibble
            or    Tmp2,Tmp1             ; add bits in upper nibble to the function
            std   Z+3,Tmp2              ; store updated function back to Tab3 Data3
            ret

Add2:       
            std   Z+2,Tmp1              ; and store it back to Tab3 Data2
            ret


;--------------------------------------------------

            ; Now the header byte will be analyzed. Therefore it is reloaded from the SRAM

CheckHeaderByte:
            lds   RxData,RxDataSRAM     ; 
            sbrc  FlagReg2,WaitQuit
            rjmp  WQuit
HBC:        andi  RxData,0b00001111     ; mask only the number of bytes of the command
            sts   NoOfDataSRAM,RxData   ; and store this number in SRAM (Number Of Data SRAM)
            sbr   FlagReg,1<<HeaderByteReceived; set Flag in FlagReg
            ret                         ; go back to main loop

WQuit:
            cbr   FlagReg2,1<<WaitQuit            
            cpi   RxData,0x20           ; if header byte is correct ...
            breq  HBC                   ; ... than proceed normally
            ldi   XH,high(RxDataSRAM)   ; restore the X-pointer to be ready for receiving new 
            ldi   XL,low(RxDataSRAM)    ; data
            sts   RufByteSRAM,RxData    ; if the received byte was not the answer from the device
                                        ; than the central did address the device again and the
                                        ; received byte should be the request for quitting again
                                        ; the new RufByte will be checked now

;- - - - - - - - - -

CheckRufByte:                           ; check parity of RufByte

            clr   Tmp1
            lds   Tmp2,RufByteSRAM      ; load RufByte from SRAM
            sbrc  Tmp2,0                ; check all bits in RufByte and count the 1s in RufByte 
            inc   Tmp1
            sbrc  Tmp2,1
            inc   Tmp1
            sbrc  Tmp2,2
            inc   Tmp1
            sbrc  Tmp2,3
            inc   Tmp1
            sbrc  Tmp2,4
            inc   Tmp1
            sbrc  Tmp2,5
            inc   Tmp1
            sbrc  Tmp2,6
            inc   Tmp1
            sbrc  Tmp2,7
            inc   Tmp1
            sbrc  Tmp1,0                ; check now, if the lowest bit of the countvalue is set
            rjmp  RufByteError          ; if it is set, than the parity-check did fail
                         
            ; if the parity is checked and the RufByte seems to be ok, than check now if it is the only
            ; byte that will be received from central and if yes, if it is valid for this slave
 
            ; if command consists only of the RufByte ... 
            cpi   Tmp2,0b01000100       ; Request       P+0x40+GA (GA = 4) ### modify parity and address
            breq  Request
            cpi   Tmp2,0b10000100       ; Quittierung   P+0x00+GA (GA = 4) ### modify parity and address
            breq  Quitting 
            cpi   Tmp2,0b00100100       ; TBD           P+0x20+GA (GA = 4) ### modify parity and address
            breq  TBD 
            
            ; if command consists of the RufByte and more data ...
            ; check now the kind of the RufByte
            cpi   Tmp2,0xa0             ; RufByte belongs to the broadcast "Rueckmeldung"
            breq  RufByteOK
            andi  Tmp2,0b01100000       ; mask only cmd-bits (RufByte was a normal command for this or any other slave)
            cpi   Tmp2,0x60             ; RufByte was a broadcast or info message for this or all devices
            breq  RufByteOk
            cpi   Tmp2,0x40             ; RufByte was a request but not for this device
            breq  SetNullDev            ;
            cpi   Tmp2,0x00             ; RufByte is Quittierung but not for this device
            breq  AskQuitting           ;
            cpi   Tmp2,0x20             ; RufByte is TBD but not for this device
            breq  SetNullDev
            
RufByteError:                           ; ### RufByte was unknown, an error alarm could be set here
            cbr   FlagReg,1<<RufByteReceived
            ret

RufByteOK:  sbr   FlagReg,1<<RufByteChecked
            ret

            ; now this device is waiting for some data but it knows that this data is not valid for itself.
            ; the data the device is now waiting for does not come from the central but from an other device.
            ; This data can be removed after it is completlty received. The only cause to receive the data
            ; is, to stay synchronious. 

SetNullDev: sbr   FlagReg,1<<RufByteChecked
            ret

            ; Central did ask for quitting an other device. This device has to listen now to the answer
            ; from the addressed device to stay synchronious.
AskQuitting:sbr   FlagReg,1<<RufByteChecked
            sbr   FlagReg2,1<<WaitQuit
            ret

;----------------------------------------------------

Request:    cbr   FlagReg,1<<RufByteReceived;
                                       ; ### normally the DataGen does not send infos back to the
                                       ; central, but in case of a detected short circuit it could send
                                       ; the cmd Alles Aus
            sbrc  FlagReg,ShortCircuitDetect
            rjmp  ShortCir             ; send information about a detected short circuit now
            rjmp  Quitting             ; send Quittung now

;----------------------------------------------------

ShortCir:   cbr   FlagReg,1<<RufByteReceived; -- Alles aus 0x21 0x80 0xA1
	    ldi   TxData,0x21      
	    sts   TxDataSRAM,TxData    ; HeaderByte
            ldi   TxData,0x80
	    sts   TxDataSRAM+1,TxData  ; Data1
            ldi   TxData,0xA1
	    sts   TxDataSRAM+2,TxData  ; XOrByte
            ldi   TxData,3
            sts   NoOfTxDataSRAM,TxData; store number of transmitted data to SRAM
            sbr   FlagReg,1<<SendTxData; set flag to indicate that data has to be send 
            cbr   FlagReg,1<<ShortCircuitDetect; clear flag again
            ret
            
;----------------------------------------------------

Quitting:   cbr   FlagReg,1<<RufByteReceived;
	    ldi   TxData,0x20      
	    sts   TxDataSRAM,TxData    ; HeaderByte
	    sts   TxDataSRAM+1,TxData  ; XOrByte
            ldi   TxData,2
            sts   NoOfTxDataSRAM,TxData; store number of transmitted data to SRAM
            sbr   FlagReg,1<<SendTxData; set flag to indicate that data has to be send 
            ret
            
;----------------------------------------------------

TBD:     
            ret

;----------------------------------------------------
;-- because a new device did now control the loco, send data to device which has
;-- controlled the loco before.

BC2OldDev:
            ;Broadcast is (P+0x60+GA 0xE3 Kennung(0x40) AH AL XOR)     
	    sts   TxDataSRAM,DeviceAdrTab3; RufByte
            ldi   XOrByte,0xE3          ; HeaderByte
            sts   TxDataSRAM+1,XOrByte
            ldi   TxData,0x40           ; Kennung
            eor   XOrByte,TxData
            sts   TxDataSRAM+2,TxData
            lds   TxData,RxDataSRAM+2   ; AH
            eor   XOrByte,TxData
            sts   TxDataSRAM+3,TxData
            lds   TxData,RxDataSRAM+3   ; AL
            eor   XOrByte,TxData
            sts   TxDataSRAM+4,TxData
            sts   TxDataSRAM+5,XOrByte  ; XOrByte
            ldi   TxData,6
            sts   NoOfTxDataSRAM,TxData ; store number of transmitted data to SRAM
            sbr   FlagReg,1<<SendTxData ; set flag to indicate that data has to be send 
            ret

;-- because the same device did control the loco, send a LI-Message now to this device
;-- any kind of message has to be send due to stay in the request-answer-scheme

BC2SameDev:
            ;Broadcast is (P+0x60+GA 0x01 0x04 XOR) (LI-Message)     
	    sts   TxDataSRAM,DeviceAdrTab3; RufByte
            ldi   XOrByte,0x01          ; HeaderByte
            sts   TxDataSRAM+1,XOrByte
            ldi   TxData,0x04           ; LI-Code
            eor   XOrByte,TxData
            sts   TxDataSRAM+2,TxData
            sts   TxDataSRAM+3,XOrByte  ; XOrByte
            ldi   TxData,4
            sts   NoOfTxDataSRAM,TxData ; store number of transmitted data to SRAM
            sbr   FlagReg,1<<SendTxData ; set flag to indicate that data has to be send 
            ret

;----------------------------------------------------
;-- this is the routine which transmitts data stored in the Tx-Data-SRAM area

TxDataSend: 
            sbis  USR,UDRE             ; Check now, if new data can be written to UDR
            ret                        ; no, than jump back to main-loop
            lds   XBusCnt,TxCntSRAM    ; load actual number of transmitted bytes from SRAM
            lds   NoOfData,NoOfTxDataSRAM; load number of bytes to be transmitted from SRAM
            cp    XBusCnt,NoOfData     ; all bytes transmitted?
            breq  TxDataEnd
            cpi   XBusCnt,0            ; is this the first byte to be transmitted?
            breq  TxDataFirst          ; yes, than jump
            sbis  USR,TXC              ; was byte completely transmitted?
            ret                        ; no, than wait
            sbi   USR,TXC              ; clear the flag that indicates that the whole byte was send
            clr   Tmp4                 ; wait now for some cycles to prevent that the SW-UARTs
            inc   Tmp4                 ; in the other modules get out of sync
            sbrs  Tmp4,5               ; this counts up to 32 0b00100000
            rjmp  PC-2
            sbi   PortD,RS485DEX       ; set the DataEnable line of the RS485 driver now
            ldi   YH,high(TxDataSRAM)  ; this only clears the upper Y byte
            ldi   YL,low(TxDataSRAM)   ; 
            add   YL,XBusCnt           ; add the actual byte position to the Y-Pointer
            ld    TxData,Y             ; load byte from SRAM
            out   UDR,TxData           ; move data into the uart data register
            inc   XBusCnt              ; increment the XBusCnt and ...
            sts   TxCntSRAM,XBusCnt    ; ... store it back to the SRAM for further operation
            ret                        ; go back to main, when finished
            
TxDataFirst:
            clr   Tmp4                 ; wait now for some cycles to prevent that the SW-UARTs
            inc   Tmp4                 ; in the other modules get out of sync
            sbrs  Tmp4,5               ; this counts up to 32 0b00100000
            rjmp  PC-2
            sbi   PortD,RS485DEX       ; set the DataEnable line of the RS485 driver now
            ldi   YH,high(TxDataSRAM)  ; this only clears the upper Y byte
            ldi   YL,low(TxDataSRAM)   ; 
            add   YL,XBusCnt           ; add the actual byte position to the Y-Pointer
            ld    TxData,Y             ; load byte from SRAM
            out   UDR,TxData           ; move data into the uart data register
            inc   XBusCnt              ; increment the XBusCnt and ...
            sts   TxCntSRAM,XBusCnt    ; ... store it back to the SRAM for further operation
            ret                        ; go back to main, when finished

TxDataEnd:  sbis  USR,TXC
	    ret
	    cbi   PortD,RS485DEX       ; clear the DataEnable line of the RS485 driver now
            cbr   FlagReg,1<<SendTxData; clear flag
            clr   XBusCnt              ; clear XBusCnt now and ...
            sts   TxCntSRAM,XBusCnt    ; ... store it back to the SRAM for further operation
            sbi   USR,TXC              ; clear the flag that indicates that the last bit was send
            ret

;----------------------------------------------------

;---------------------------------------------------------------------------------------------------- 
;-- Subroutine which controls a short circuit on the track
;-- 
;-- 
;---------------------------------------------------------------------------------------------------- 
EXT_INT0:
            PUSHTmp
            cbi   PortD,RelaisEn        ; switch off the signal to the booster
            sbr   FlagReg,1<<ShortCircuitDetect
            POPTmp
            reti

;---------------------------------------------------------------------------------------------------- 
;-- Subroutine which stores a byte received from the X-Bus to the SRAM  
;-- 
;-- 
;---------------------------------------------------------------------------------------------------- 

UART_RXC:
        PUSHTmp
        in      RxData,UDR              ; read byte from data-receive-register 
        sbr     FlagReg,1<<XDataReceived; set flag to indicate that a new byte was received from the x-bus
        sbrs    FlagReg,RufByteReceived ; was RufByte just received?
        rjmp    StoreRufByte            ; no, than save actual received byte as RufByte in SRAM
        st      X+,RxData               ; save new received byte in SRAM and incremet the WrPointer
        POPTmp
        reti

StoreRufByte:
        sts     RufByteSRAM,RxData
        sbr     FlagReg,1<<RufByteReceived; set RufByte-flag
        POPTmp
        reti

;-@-----------------------------------------------------------------------------------------------------
;-- in der folgenden (Interrupt)-Routine werden die korekten Zeiten fuer das Bitmuster zum Booster
;-- generiert. Ausserdem wird in der Pause T3 bestimmt, von welchem Fahrpult welche Information als
;-- naechstes zu senden ist. Diese Informationen werden beim ersten Aufruf nach der Pause T3 aus dem
;-- SRAM ausgelesen
;-------------------------------------------------------------------------------------------------------

.include "datagen_t1_compb.asm"

;---------------------------------------------------------------------------------------------------- 
;-- Konstanten-Tabellen
;---------------------------------------------------------------------------------------------------- 

.cseg
.org 0x0010

SpeedTable:
;   Bit  7654321076543210
   .DW 0b0001000101000101; + 0, - 0  11 45
   .DW 0b0011000101100101; + 1, - 1  31 65
   .DW 0b1011000111100101; + 2, - 2  b1 e5
   .DW 0b0001100101001101; + 3, - 3  19 4d
   .DW 0b1001100111001101; + 4, - 4  99 cd
   .DW 0b0011100101101101; + 5, - 5  39 6d
   .DW 0b1011100111101101; + 6, - 6  b9 ed
   .DW 0b0001001001000110; + 7, - 7  12 46
   .DW 0b1001001011000110; + 8, - 8  92 c6
   .DW 0b0011001001100110; + 9, - 9  32 66
   .DW 0b1011001011100110; +10, -10  b2 e6
   .DW 0b0001101001001110; +11, -11  1a 4e
   .DW 0b1001101011001110; +12, -12  9a ce
   .DW 0b0011101001101110; +13, -13  3a 6e
   .DW 0b1011101011101110; +14, -14  ba ee
   .DW 0b1001000111000101; + r, - r  91 c5

;--------------------------------------------------------------------------------
.cseg 
.org 0x0020

;   lpm-Kommando liest Low-Byte,  dann wird inkrementiert
;   lpm-Kommando liest High-Byte, dann wird inkrementiert
;   lpm-Kommando liest Low-Byte,  dann wird inkrementiert
;   lpm-Kommando liest High-Byte

FunctionTable:
;         F2(off) F1(off)
;         F4(off) F3(off)
;   Bit  7654321076543210
   .DW 0b0000010001010000; speed  0  01 50
   .DW 0b0101010000010100; speed  0  54 14
   .DW 0b0010010001110000; speed  1  24 70
   .DW 0b0111010000110100; speed  1  74 34
   .DW 0b1010010011100100; speed  2  a4 e4
   .DW 0b1111010010110100; speed  2  f4 b4
   .DW 0b0100110001011000; speed  3  4c 58
   .DW 0b0101110000011100; speed  3  5c 1c
   .DW 0b1000110011011000; speed  4  8c d8
   .DW 0b1101110010011100; speed  4  dc 9c
   .DW 0b0010110001111000; speed  5  2c 78
   .DW 0b0111110001101100; speed  5  7c 6c
   .DW 0b1010110011111000; speed  6  ac f8
   .DW 0b1110110010111100; speed  6  ec bc
   .DW 0b0000011001010010; speed  7  06 52
   .DW 0b0101011000010110; speed  7  56 16
   .DW 0b1000011011010010; speed  8  86 d2
   .DW 0b1101011010010110; speed  8  d6 96
   .DW 0b0010011001110010; speed  9  26 72
   .DW 0b0111011000110110; speed  9  76 36
   .DW 0b1010011011110010; speed 10  a6 f2
   .DW 0b1111011010110110; speed 10  f6 b6
   .DW 0b0000111001011010; speed 11  0e 5a
   .DW 0b0101111000011110; speed 11  4e 1e
   .DW 0b1000111011011010; speed 12  8e da
   .DW 0b1101111010011110; speed 12  de 9e
   .DW 0b0010111001111010; speed 13  2e 7a
   .DW 0b0111111000111110; speed 13  7e 3e
   .DW 0b1010111011111010; speed 14  ae fa
   .DW 0b1111111010111110; speed 14  fe be
   .DW 0b1000010011010000; speed  r  84 d0
   .DW 0b1101010010010100; speed  r  d4 94
		 
;--------------------------------------------------------------------------------

.cseg
.org 0x0040

AddressTable:          
;  Byte  | High || Low  |
;  Bit   7654321076543210

   .DW 0b1100000010101010; C0 AA   1 	0   Adresse 0 entspricht dem Idle-Pattern, muesste aber eigentlich 00000000 sein
   .DW 0b0011000010000000; 30 80   3 	2
   .DW 0b1011000011110000; B0 F0   5 	4
   .DW 0b1110000000100000; E0 20   7 	6
   .DW 0b0000110010100000; 0C A0   9 	8
   .DW 0b1000110011001100; 8C CC  11   10
   .DW 0b1111110000111100; FC 3C  13   12
   .DW 0b0010110010111100; 2C BC  15   14
   .DW 0b1010110011101100; AC EC  17   16
   .DW 0b1100100000001000; C8 08  19   18
   .DW 0b0011100010001000; 38 88  21   20
   .DW 0b1011100011111000; B8 F8  23   22
   .DW 0b1110100000101000; E8 28  25   24
   .DW 0b0000001110101000; 03 A8  27   26
   .DW 0b1000001111000011; 83 C3  29   28
   .DW 0b1111001100110011; F3 33  31   30
   .DW 0b0010001110110011; 23 B3  33   32
   .DW 0b1010001111100011; A3 E3  35   34
   .DW 0b1100111100001111; CF 0F  37   36
   .DW 0b0011111110001111; 3F 8F  39   38
   .DW 0b1011111111111111; BF FF  41   40
   .DW 0b1110111100101111; EF 2F  43   42
   .DW 0b0000101110101111; 0B AF  45   44
   .DW 0b1000101111001011; 8B CB  47   46
   .DW 0b1111101100111011; FB 3B  49   48
   .DW 0b0010101110111011; 2B BB  51   50
   .DW 0b1010101111101011; AB EB  53   52
   .DW 0b1100001000000010; C2 02  55   54
   .DW 0b0011001010000010; 32 82  57   56
   .DW 0b1011001011110010; B2 F2  59   58
   .DW 0b1110001000100010; E2 22  61   60
   .DW 0b0000111010100010; 0E A2  63   62
   .DW 0b1000111011001110; 8E CE  65   64
   .DW 0b1111111000111110; FE 3E  67   66
   .DW 0b0010111010111110; 2E BE  69   68
   .DW 0b1010111011101110; AE EE  71   70
   .DW 0b1100101000001010; CA 0A  73   72
   .DW 0b0011101010001010; 3A 8A  75   74
   .DW 0b1011101011111010; BA FA  77   76
   .DW 0b1110101000101010; EA 2A  79   78
   .DW 0b0100000000000000; 40 00  81   80   Adresse 80 := Idle-Pattern, muesste also 10101010 sein
   .DW 0b1001011101100000; 97 60  83   82 
   .DW 0b0100100001110000; 48 70  85   84 
   .DW 0b0101100001101000; 58 68  87   86 
   .DW 0b0100010001111000; 44 78  89   88 
   .DW 0b0101010001100100; 54 64  91   90 
   .DW 0b0100110001110100; 4C 74  93   92 
   .DW 0b0101110001101100; 5C 6C  95   94 
   .DW 0b0100001001111100; 42 7C  97   96 
   .DW 0b0101001001100010; 52 62  99   98 
   .DW 0b0100101001110010; 4A 72  101 100
   .DW 0b0101101001101010; 5A 6A  103 102
   .DW 0b0100011001111010; 46 7A  105 104
   .DW 0b0101011001100110; 56 66  107 106
   .DW 0b0100111001110110; 4E 76  109 108
   .DW 0b0101111001101110; 5E 6E  111 110
   .DW 0b0100000101111110; 41 7E  113 112
   .DW 0b0101000101100001; 51 61  115 114
   .DW 0b0100100101110001; 49 71  117 116
   .DW 0b0101100101101001; 59 69  119 118
   .DW 0b0100010101111001; 45 79  121 120
   .DW 0b1001111101100101; 9F 65  123 122
   .DW 0b0100110101110101; 4D 75  125 124
   .DW 0b0101110101101101; 5D 6D  127 126
   .DW 0b0100001101111101; 43 7D  129 128
   .DW 0b0101001101100011; 53 63  131 130
   .DW 0b0100101101110011; 4B 73  133 132
   .DW 0b0101101101101011; 5B 6B  135 134
   .DW 0b0100011101111011; 47 7B  137 136
   .DW 0b0101011101100111; 57 67  139 138
   .DW 0b0100111101110111; 4F 77  141 140
   .DW 0b0101111101101111; 5F 6F  143 142
   .DW 0b0001000001111111; 10 7F  145 144
   .DW 0b0001010000011000; 14 18  147 146
   .DW 0b0001001000011100; 12 1C  149 148
   .DW 0b0001011000011010; 16 1A  151 150
   .DW 0b0001000100011110; 11 1E  153 152
   .DW 0b0001010100011001; 15 19  155 154
   .DW 0b0001001100011101; 13 1D  157 156
   .DW 0b0001011100011011; 17 1B  159 158
   .DW 0b1101000000011111; D0 1F  161 160
   .DW 0b1101010011011000; D4 D8  163 162
   .DW 0b1101001011011100; D2 DC  165 164
   .DW 0b1101011011011010; D6 DA  167 166
   .DW 0b1101000111011110; D1 DE  169 168
   .DW 0b1101010111011001; D5 D9  171 170
   .DW 0b1101001111011101; D3 DD  173 172
   .DW 0b1101011111011011; D7 DB  175 174
   .DW 0b1001000011011111; 90 DF  177 176
   .DW 0b1001010010011000; 94 98  179 178
   .DW 0b1001001010011100; 92 9C  181 180
   .DW 0b1001011010011010; 96 9A  183 182
   .DW 0b1001000110011110; 91 9E  185 184
   .DW 0b1001010110011001; 95 99  187 186
   .DW 0b1001001110011101; 93 9D  189 188
   .DW 0b0101000010011011; 50 9B  191 190
   .DW 0b0000010001010101; 04 55  193 192
   .DW 0b0000010100000110; 05 06  195 194
   .DW 0b1100010000000111; C4 07  197 196
   .DW 0b1100010111000110; C5 C6  199 198
   .DW 0b1000010011000111; 84 C7  201 200
   .DW 0b1000010110000110; 85 86  203 202
   .DW 0b0011010010000111; 34 87  205 204
   .DW 0b0011010100110110; 35 36  207 206
   .DW 0b1111010000110111; F4 37  209 208
   .DW 0b1111010111110110; F5 F6  211 210
   .DW 0b1011010011110111; B4 F7  213 212
   .DW 0b1011010110110110; B5 B6  215 214
   .DW 0b0010010010110111; 24 B7  217 216
   .DW 0b0010010100100110; 25 26  219 218
   .DW 0b1110010000100111; E4 27  221 220
   .DW 0b1110010111100110; E5 E6  223 222
   .DW 0b1010010011100111; A4 E7  225 224
   .DW 0b1010010110100110; A5 A6  227 226
   .DW 0b0000000110100111; 01 A7  229 228
   .DW 0b1000000111000001; 81 C1  231 230
   .DW 0b1111000100110001; F1 31  233 232
   .DW 0b0010000110110001; 21 B1  235 234
   .DW 0b1010000111100001; A1 E1  237 236
   .DW 0b1100110100001101; CD 0D  239 238
   .DW 0b0011110110001101; 3D 8D  241 240
   .DW 0b1011110111111101; BD FD  243 242
   .DW 0b1110110100101101; ED 2D  245 244
   .DW 0b0000100110101101; 09 AD  247 246
   .DW 0b1000100111001001; 89 C9  249 248
   .DW 0b1111100100111001; F9 39  251 250
   .DW 0b0010100110111001; 29 B9  253 252
   .DW 0b1010100111101001; A9 E9  255 254
